# minimalistic implementation of basic operations on mod p elliptic curves
#
# gabriel.chenevert@junia.com 2025/11


from is_prime import *


large = 2**10  # arbitrary


def div_mod(a, b, p):
	
	return a * pow(b,-1,p) % p


# courbe elliptique mod p sous forme de Weierstrass

class EllipticCurve:
	
	def __init__(self, p, a, b, trust = False):

		if not trust:  # check parameters

			if not is_prime(p) or p == 2:

				raise ValueError("Only odd prime moduli are supported")
				
			if (4*a**3 + 27*b*2) % p == 0:
				
				raise ValueError("Curve is singular")
			
		self.p = p
		self.a = a % p
		self.b = b % p
		
	def __repr__(self):
		
		return "Elliptic curve y^2 = x^3 + %sx + %s mod %s" % (self.a, self.b, self.p)
		
	def homog_def_equation(self, x, y, z = 1):
		
		return (y**2*z - x**3 - self.a*x*z**2 - self.b*z**3) % self.p
		
	def point(self, x, y, z = 1, trust = False):
	
		return EllipticPoint(self, x, y, z, trust)
		
	def zero(self):
		
		return EllipticPoint(self, 0, 1, 0, True)
		
	def add(self, P, Q):
		
		if P.E != self or Q.E != self:
			
			raise ValueError("Points must be on same curve")
			
		if P == self.zero():
			
			return Q
			
		if Q == self.zero():
			
			return P
		
		if P.x == Q.x:
			
			if P != Q:  # vertically aligned
				
				return self.zero()
				
			# doubling
			
			if P.y == 0:
				
				return self.zero()
			
			l = div_mod(3*P.x**2 + self.a, 2*P.y, self.p)
			
		else:  # generic case
			
			l = div_mod(Q.y - P.y, Q.x - P.x, self.p)
		
		x = l**2 - (P.x + Q.x)
		y = l * (x - P.x) + P.y
			
		return EllipticPoint(self, x, -y, True)
		
	def mul(self, m, P):
		
		res = self.zero()
		
		if m < 0:
			
			P = P.opp()
			m = -m
		
		for b in bin(m)[2:]:
			
			res = res + res
			if int(b):
				res = res + P
				
		return res
		
	def all_points(self):
		
		if self.p > large:
			
			raise ValueError("Curve too large for reasonable exhaustive search")
		
		pts = [self.zero()]
		
		# pourrait être un peu moins brutal
		
		for x in range(self.p):
			for y in range(self.p):
				
				if self.homog_def_equation(x,y) == 0:
					pts.append(EllipticPoint(self, x, y, True))
					
		return pts


# point sur une courbe
		
class EllipticPoint:
	
	def __init__(self, E, x, y, z = 1, trust = False):

		if not trust:
			
			if E.homog_def_equation(x,y,z) != 0:
			
				raise ValueError("Point not on curve")
			
			if z % E.p != 0:
				
				x = div_mod(x, z, E.p)
				y = div_mod(y, z, E.p)
				z = 1
				
			else:
				
				x = 0
				y = 1
			
		self.E = E
		self.x = x % E.p
		self.y = y % E.p
		self.z = z % E.p  # assumed 0 or 1
		
	def __eq__(self, other):
	
		return (self.x, self.y, self.z) == (other.x, other.y, other.z)
		
	def __repr__(self):
		
		if self.z == 0:
			return "0"
		if self.E.p < large:
			return "(%s,%s)" % (self.x, self.y)
		else:
			return "(%s,\n %s)" % (hex(self.x), hex(self.y))
			
	def __add__(self, other):
		
		return self.E.add(self, other)
		
	def __sub__(self, other):
		
		return self + other.opp()
		
	def opp(self):
		
		return EllipticPoint(self.E, self.x, -self.y, self.z, True)
		
	def __rmul__(self, m):
		
		return self.E.mul(m, self)
